function [out,fmin,parout] =fista(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,par)
%FISTA employs the fista method for solving the problem min{f(x) + lambda* g(x)}
%
% Underlying assumptions:
% All functions are convex
% f is smooth
% g is proper closed and proximable
% lambda is a positive scalar
% ====================================================
% Usage:
% out               = FISTA(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,[par])
% [out,fmin]        = FISTA(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,[par])
% [out,fmin,parout] = FISTA(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,[par])
% ====================================================
% Input:
% Ffun        - function handle for the function f
% Ffun_grad   - function handle for the gradient of the function f
% Gfun        - function handle for the function g
% Gfun_prox   - function handle for the proximal mapping of g times a postive constant
% lambda      - positive scalar penalty for the function g
% startx      - starting vector
% par         - struct which contains different values required for the operation of FISTA
% Fields of par:
%       max_iter      - maximal number of iterations [default: 1000]
%       eco_flag      - true if economic version (without calculating objective function
%                              values) should run, otherwise false [default: false]
%       print_flag    - true if internal printing should take place, otherwise false [default: true]
%       monotone_flag - true if monotonic method should be applied, otherwise false [default: false]
%       Lstart        - initial value for the Lipschitz constant (scalar) [default: 1]
%       const_flag    - true if constant step size should be used, otherwise false [default: false]
%       regret_flag   - true if the Lipschitz constant should be divided by eta in the next iteration [default: false]
%       eta           - the multiplicative constant used when regretting or backtracking [default: 2]
%       eps           - stopping criteria tolerance (the method stops when the
%                       norm of the difference between consecutive iterates is < eps) [default: 1e-5]
%                       relevant only in non monotonic method
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    LValVec   -  vector of all used Lipschitz constants (when par.const_flag is false) 

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

if ((nargin < 6) ||  ~isa (Ffun_grad,'function_handle') || ~isa (Gfun_prox,'function_handle') )
    error ('usage: fista(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,[par])) ') ;
end

%setting default values

if (nargin < 7)
    par = struct() ;
end

if ( isempty(Ffun) || isempty(Gfun) || ~isa(Ffun,'function_handle') || ~isa(Gfun,'function_handle') )
    if (nargout > 1)
        error('When FFun or Gfun is not a function handle usage: out = fista([],Ffun_grad,[],Gfun_prox,lambda,startx,[par])')
    else
        if ( ~isfield (par,'eco_flag') || (~par.eco_flag))
            fprintf('Ffun or Gfun is not a function handle, changing to economical mode\n') ;
            par.eco_flag = true ;
            par.print_flag = false ;
            par.const_flag = true ;
            par.monotone_flag = false ;
        end
    end
end

par = initialize_par_for_prox_gradient(par,'fista') ;


%constant values
Lstart = par.Lstart ;
const_flag = par.const_flag ;
regret_flag = par.regret_flag ;
max_iter = par.max_iter ;
eco_flag = par.eco_flag ;
print_flag = par.print_flag ;
monotone_flag = par.monotone_flag ;
eta = par.eta ;
if ( ~monotone_flag)
    eps = par.eps ;
end

if (nargout == 3)
    parout = struct ;
    parout.iterNum = 0 ;
    parout.funValVec = [] ;
    if (~const_flag)
        parout.LValVec = [] ;
    end
end


%Make first vector feasible
startx = Gfun_prox(startx,1e-4);

Feps = 1e-10 ; %Inner epsilon for comparing newFvalue and quad_aprrox

%starting values
iternu=0; %iteration number
prevx = startx ;
newx = prevx ; 
newy = startx ;
prevt = 1 ;
done = false ;

if (~eco_flag)
    prevFvalue = Ffun(startx) ;
    prevGvalue = Gfun(startx) ;
    best_vector = startx ;
    best_value = prevFvalue + lambda * prevGvalue  ; %best value for objective function found so far
end


%starting the L according to const_flag and regret_flag
if (const_flag)
    L = Lstart ;
else
    if (regret_flag)
        L = Lstart * eta ; % in the first iteration it will be divided by eta
    else
        L = Lstart ;
    end
end
 
%First printing
if(print_flag)
    fprintf('*********************\n');
    fprintf('FISTA\n') ;
    fprintf('*********************\n')
    if (~const_flag)
        fprintf('#iter       \tfun. val.     \tL val.\n');
    else
        fprintf('#iter       \tfun. val.\n');
    end
    
    
end



%the iterations
while((iternu < max_iter) && (~done))  
    iternu = iternu + 1;
    
    if ((~const_flag) && (regret_flag))
        L = L / eta ;
    end

    grad = Ffun_grad(newy) ;
    prox_val = Gfun_prox(newy-1/L*grad, lambda/L) ;
    if (~eco_flag)
        newFvalue = Ffun(prox_val) ;
    end
    
    if (~const_flag)
        %search for appropriate L
        FvalY = Ffun(newy) ;
        quad_approx = FvalY + trace(grad'*(prox_val-newy)) + L/2*norm(prox_val-newy,'fro')^2 ;
        while (newFvalue  > quad_approx + Feps) ;
            L= L * eta ;
            prox_val = Gfun_prox(newy-1/L*grad, lambda/L) ;
            newFvalue = Ffun(prox_val) ;
            quad_approx = FvalY + trace(grad'*(prox_val-newy)) + L/2*norm(prox_val-newy,'fro')^2 ;
        end
    end
    
    newt = 0.5* (1+sqrt(1+4*prevt^2)) ;
    if (~eco_flag)
        newGvalue = Gfun(prox_val) ;
    end
        
    if (monotone_flag)
        newz = prox_val ;
        if (newFvalue + lambda * newGvalue< prevFvalue + lambda*prevGvalue)
            newx = prox_val ; %newz
        else
            newFvalue = prevFvalue ;
            newGvalue = prevGvalue ;
        end
        
        newy = newx + prevt /newt *(newz-newx) + (prevt -1)/newt *(newx-prevx) ;
        prevx = newx ;
        prevFvalue = newFvalue ;
        prevGvalue = newGvalue ;
    else
        newx = prox_val ;
        if (norm(newx - prevx,'fro') < eps)
            done = true ;
        end
     
        
        newy = newx + (prevt -1)/newt *(newx-prevx) ;
        prevx = newx ;
    end
    
    prevt = newt ;
    if (~eco_flag)
        newFGvalue = newFvalue+lambda*newGvalue ;
        if (nargout  == 3)
            %updating parout
            parout.funValVec = [ parout.funValVec ; newFGvalue] ;
            if (~const_flag)
                parout.LValVec = [parout.LValVec ; L] ;
            end
        end
        
        if (monotone_flag || (newFGvalue < best_value))
            best_vector = newx ;
            best_value = newFGvalue ;
            
            if (print_flag)
                fprintf('%6d \t    %12f \t',iternu,newFGvalue);
                if (~const_flag)
                    fprintf('%12f',L) ;
                end            
                fprintf('\n') ;
            end
        end
    end
    
    
end


if (done && print_flag)
    fprintf('Stopping because the norm of the difference between consecutive iterates is too small\n')
end

if (eco_flag)
    out = newx ;
else
    out = best_vector ;
end

if ((nargout >1) || (print_flag))
    if (~eco_flag)
        fmin = best_value ;
    else
        fmin = Ffun(out)+lambda*Gfun(out) ;
    end     
end

if (nargout == 3)
    %updating parout.iterNum
    parout.iterNum = iternu ;
end

if (print_flag)
    fprintf('----------------------------------\n') ;
    fprintf('Optimal value = %15f \n',fmin) ;
end



